# Deep Refactoring Workflow

A comprehensive multi-step refactoring workflow that systematically identifies duplications, plans improvements, and implements changes with continuous validation.

## What It Does

This workflow performs a complete refactoring cycle with **incremental validation**:

1. **Analyze Code Structure** - Maps all key symbols, architectural patterns, and component relationships
2. **Find Explicit Duplications** - Identifies copy-pasted code and repeated patterns
3. **Find Implicit Duplications** - Discovers semantic similarities that should be unified
4. **Create Refactoring Plan** - Builds a prioritized plan saved to `.refactoring-plan.md`
5. **Execute Refactorings Loop** - Calls the internal iteration workflow
6. **Save Summary** - Documents all improvements in `refactoring-summary.md`
7. **Cleanup** - Removes temporary plan file

### Internal Loop (Deep Refactoring Iteration)

The workflow uses an internal loop that:
1. **Reads the plan** from `.refactoring-plan.md`
2. **Takes the first unchecked refactoring** from the list
3. **Implements it** and marks as complete
4. **Runs Tests Must Pass** to validate the change
5. **Calls itself recursively** if more refactorings remain

This ensures **each refactoring is tested individually** before moving to the next one.

## Key Features

### Analytical Modes
Uses specialized modes for different phases:
- `SC:Architect` - For structural analysis and planning
- `SC:Refactor` - For finding duplications and optimization opportunities

### Incremental Validation
**Key difference from linear workflows:** Tests run after EACH individual refactoring, not once at the end.

The internal loop ensures:
- Implements one refactoring at a time
- Runs "Tests Must Pass" immediately after each change
- Only proceeds to next refactoring if tests pass
- No regressions accumulate across multiple changes
- Easy to identify which refactoring caused issues (if any)

### Comprehensive Output
Creates `refactoring-summary.md` with:
- Executive summary of improvements
- Complete list of changes made
- Architecture improvements achieved
- Testing and validation results
- Next steps if needed

## Usage

### When to Use

**Good for:**
- Codebases with accumulated technical debt
- Projects with noticeable code duplication
- Before major feature additions (clean foundation)
- After rapid prototyping phase (consolidation)
- Regular maintenance cycles

**Best results when:**
- You have a test suite in place
- The codebase is under version control
- You want systematic rather than ad-hoc refactoring

### How to Use

1. Ensure you have tests for critical functionality
2. Click the "Deep Refactoring" button
3. The workflow will analyze, plan, and execute refactorings
4. Review `refactoring-summary.md` when complete

### Estimated Time

Depends on codebase size:
- Small projects (<50 files): 10-20 minutes
- Medium projects (50-200 files): 30-60 minutes
- Large projects (200+ files): 1-2 hours

## Output Format

The workflow maintains concise, actionable outputs at each stage (1-2 sentences per finding) to keep the process focused and efficient.

Final `refactoring-summary.md` includes:

```markdown
# Refactoring Summary

## Executive Summary
Brief overview of what was refactored and key improvements.

## Changes Made
- Refactoring 1: Description, files affected, impact
- Refactoring 2: Description, files affected, impact
...

## Architecture Improvements
- Duplications eliminated
- New abstractions introduced
- Code organization improvements

## Testing & Validation
- All tests passing: ✓
- New tests added: [count]
- Validation performed: [description]

## Next Steps
[Any remaining refactorings or recommendations]
```

## Examples

### Example 1: React Component Duplication

**Before:**
```
src/
  UserCard.tsx        // 120 lines
  ProductCard.tsx     // 115 lines (95% similar)
  OrderCard.tsx       // 118 lines (93% similar)
```

**After workflow:**
```
src/
  components/
    Card.tsx          // 80 lines (generic component)
  UserCard.tsx        // 35 lines (uses Card)
  ProductCard.tsx     // 32 lines (uses Card)
  OrderCard.tsx       // 34 lines (uses Card)
```

**Result:** 120 lines → 181 lines total, but with shared abstraction and easier maintenance.

### Example 2: Utility Function Consolidation

**Before:**
```typescript
// Scattered across 5 files
function formatDate1(date) { ... }
function formatDate2(date) { ... }
function dateFormatter(date) { ... }
```

**After workflow:**
```typescript
// utils/date.ts
export function formatDate(date: Date): string { ... }
```

**Result:** 3 implementations → 1 unified, type-safe function.

## Workflow Structure

The implementation uses two workflows:

```yaml
Deep Refactoring:                    # Main workflow with button
  icon: 🏗️
  menu: buttons
  actions:
    - Analyze Code Structure         # SC:Architect mode
    - Find Explicit Duplications     # SC:Refactor mode
    - Find Implicit Duplications     # SC:Refactor mode
    - Create Refactoring Plan        # Save to .refactoring-plan.md
    - Execute Refactorings Loop      # Call Deep Refactoring Iteration
    - Save Refactoring Summary       # Create refactoring-summary.md
    - Cleanup                        # Remove plan file

Deep Refactoring Iteration:          # Inner loop (no external trigger)
  actions:
    - Read Current Plan              # cat .refactoring-plan.md
    - Check if More Refactorings     # grep for unchecked items
      if: unchecked items exist
      actions:
        - Implement Next Refactoring # Take first [ ], implement, mark [x]
          actions:
            - Tests Must Pass        # Validate this refactoring
            - Deep Refactoring Iteration  # Loop: next refactoring
```

**Plan file format (`.refactoring-plan.md`):**
```markdown
# Refactoring Plan

- [ ] Extract UserCard/ProductCard common logic to Card component, files: src/UserCard.tsx, src/ProductCard.tsx
- [ ] Consolidate date formatting utilities, files: src/utils/date1.ts, src/utils/date2.ts
- [x] Remove duplicate validation logic, files: src/forms/LoginForm.tsx, src/forms/SignupForm.tsx
```

The `[ ]` checkbox syntax allows the workflow to track which refactorings are complete and which remain.

## Tips

1. **Start with tests** - Ensure you have basic test coverage before refactoring
2. **Review the plan** - The workflow shows the plan before implementing
3. **Incremental commits** - Consider committing after each validated change
4. **Monitor iterations** - Watch for any test failures during implementation
5. **Read the summary** - `refactoring-summary.md` provides valuable architecture insights

## Customization

### Adjusting Analysis Depth

Edit the workflow to make analysis more/less detailed:

```yaml
- name: 🔍 Analyze Code Structure
  prompt: |
    Analyze the codebase focusing on [specific area].
    # Add constraints or focus areas here
```

### Skipping Tests

If you don't have tests yet (not recommended), remove the Tests Must Pass step:

```yaml
- name: 🔧 Implement Refactorings
  prompt: |
    Execute the refactoring plan...
  instantRun: true
  # Remove this:
  # actions:
  #   - 'Tests Must Pass'
```

### Adding Custom Validation

Add additional validation steps:

```yaml
- name: 🔧 Implement Refactorings
  instantRun: true
  actions:
    - 'Tests Must Pass'
    - name: Run Linter
      prompt:
        command: 'npm run lint'
```

## Benefits

- **Systematic approach** - No random refactoring, everything is planned
- **Safety first** - Continuous validation prevents regressions
- **Documentation** - Automatic summary of improvements
- **Time saving** - Automated detection of duplications you might miss
- **Architectural insights** - Deep analysis reveals patterns and opportunities

---

*Clean code is not written by following a set of rules. Clean code is written by systematically eliminating duplication.*
