# Ralph Wiggum Loop Workflow

> "I'm helping!" - Ralph Wiggum

This is a port of the iconic [Ralph Wiggum technique](https://awesomeclaude.ai/ralph-wiggum) from Claude Code - an iterative AI development methodology that embodies persistent iteration despite setbacks.

## Philosophy

The Ralph Wiggum technique is based on four core principles:

### 1. Iteration > Perfection
Don't aim for perfect on first try. Let the loop refine the work.

### 2. Failures Are Data
Deterministically bad means failures are predictable and informative.

### 3. Operator Skill Matters
Success depends on writing good prompts, not just having a good model. **LLMs are mirrors of operator skill.**

### 4. Persistence Wins
Keep trying until success. The loop handles retry logic automatically.

## How It Works

The classic Ralph Wiggum technique in Claude Code:

```bash
while :; do cat PROMPT.md | claude ; done
```

In Supercode Workflows, this becomes a controlled loop with safety mechanisms:

**Main Workflow (Ralph Wiggum Loop)** - External trigger:
1. Initialize iteration counter
2. Set Ralph Mode system prompt
3. Call the iteration loop
4. Cleanup counter when done

**Inner Workflow (Ralph Iteration)** - The actual loop:
1. Increment counter
2. Work on task with iteration number
3. Check completion conditions
4. If not done - call itself recursively

The loop continues until: completion signal, stuck signal, or max iterations (20).

## The Critical Skill: Prompt Writing

Success with Ralph depends on **writing good prompts**. Here are the best practices:

### ❌ Bad Prompt
```
Build a todo API and make it good.
```

### ✅ Good Prompt
```
Build a REST API for todos.

When complete:
- All CRUD endpoints working
- Input validation in place
- Tests passing (coverage > 80%)
- README with API docs

Output: <promise>COMPLETE</promise>
```

## Prompt Writing Best Practices

### 1. Clear Completion Criteria

Always specify exactly what "done" means and include the completion signal.

```
Implement user authentication system.

Completion criteria:
- JWT token generation and validation
- Login/logout endpoints
- Password hashing with bcrypt
- Auth middleware for protected routes
- Unit tests for all functions
- Integration tests for endpoints

Output: <promise>COMPLETE</promise>
```

### 2. Incremental Goals

Break complex tasks into phases:

```
Build e-commerce checkout flow:

Phase 1: Shopping cart
- Add/remove items
- Update quantities
- Calculate totals
- Tests for cart operations

Phase 2: Checkout process
- Shipping address form
- Payment integration
- Order confirmation
- Tests for checkout flow

Phase 3: Order management
- Order history page
- Order status tracking
- Email notifications
- Tests for order operations

Output <promise>COMPLETE</promise> when all phases done.
```

### 3. Self-Correction Pattern

Guide the AI to test and fix issues:

```
Implement feature X following TDD:

1. Write failing tests for the feature
2. Implement the feature
3. Run the tests
4. If any fail, debug and fix
5. Refactor if needed
6. Repeat steps 3-5 until all tests green
7. Verify code quality (linting, formatting)

Output: <promise>COMPLETE</promise>
```

## Usage

### Basic Usage

1. Write your task with clear completion criteria (see examples above)
2. Include `<promise>COMPLETE</promise>` in your prompt as the completion signal
3. Click the "Ralph Wiggum Loop" button
4. Watch Ralph iterate until completion

### When to Use Ralph

**Good for:**
- Complex multi-step tasks requiring iteration
- Tasks where the first attempt might need refinement
- TDD workflows (test, implement, fix, repeat)
- Exploratory development
- Tasks with self-verification steps

**Not ideal for:**
- Simple one-shot tasks
- Tasks requiring careful upfront planning
- Tasks that need human decision-making between steps

## Safety Mechanisms

### Max Iterations
The workflow has a built-in limit of **20 iterations**. This prevents infinite loops on impossible tasks.

### Stuck Detection
If the AI realizes it's stuck, it can output:
```
<promise>STUCK</promise>
```

This will stop the loop and the AI will explain what's blocking progress.

### Iteration Counter
Each iteration shows the current count, helping you monitor progress:
```
(Iteration: 5)
```

## Customization

### Changing Max Iterations

Edit the `Ralph Iteration` workflow to adjust the limit:

```yaml
Ralph Iteration:
  actions:
    # ... other steps
    - name: 🔁 Loop Again
      if:
        type: js
        value: |
          (() => {
            const maxIterations = 50;  // Change this value
            // ...
          })()
```

### Custom Completion Signals

Add more completion signals in the condition:

```yaml
const isComplete = response.includes('<promise>COMPLETE</promise>')
                || response.includes('<promise>ALL_DONE</promise>');
```

## Example Prompts

### Example 1: API Development
```
Create a RESTful API for a blog system.

Requirements:
- POST /posts - Create post (title, content, author)
- GET /posts - List all posts
- GET /posts/:id - Get single post
- PUT /posts/:id - Update post
- DELETE /posts/:id - Delete post
- Input validation for all endpoints
- Error handling with proper HTTP codes
- Unit tests for all operations
- Integration tests for all endpoints
- Test coverage > 80%

After each step, run tests and fix any failures.
Output: <promise>COMPLETE</promise>
```

### Example 2: Frontend Component
```
Build a reusable data table component:

Phase 1: Basic table
- Render data from props
- Column headers
- Responsive design

Phase 2: Features
- Sorting by column
- Filtering
- Pagination

Phase 3: Quality
- TypeScript types
- Unit tests (Jest)
- Storybook stories
- Accessibility (ARIA labels)

Test after each phase. Fix issues before next phase.
Output: <promise>COMPLETE</promise>
```

### Example 3: TDD Pattern
```
Implement a user registration form with TDD:

1. Write tests for form validation (email, password strength, matching passwords)
2. Implement the validation logic
3. Run tests - if any fail, debug and fix
4. Write tests for form submission
5. Implement submission handler
6. Run tests - if any fail, debug and fix
7. Write tests for error handling (network errors, duplicate email)
8. Implement error handling
9. Run tests - if any fail, debug and fix
10. Verify all tests pass and coverage > 90%

Output: <promise>COMPLETE</promise>
```

## Tips for Success

1. **Be specific** - Vague prompts get vague results
2. **Include the completion signal** - Always add `<promise>COMPLETE</promise>` to your prompt
3. **Set clear criteria** - Define what "done" looks like
4. **Break it down** - Use phases for complex tasks
5. **Include testing** - Ask for tests and verification
6. **Trust the loop** - Don't interrupt too early
7. **Learn from iterations** - Watch how the AI improves with each pass

## Workflow Structure

The implementation uses two workflows:

```yaml
Ralph Wiggum Loop:           # Main workflow with button
  icon: 🔄
  menu: buttons
  actions:
    - Initialize Counter      # Set counter to 0
    - Activate Ralph Mode     # Set system prompt
    - Start Iteration Loop    # Call Ralph Iteration
    - Cleanup                 # Remove counter file

Ralph Iteration:             # Inner loop (no external trigger)
  actions:
    - Increment Counter       # counter++
    - Work on Task            # AI works on $initialPrompt
    - Loop Again              # If not done, call Ralph Iteration
```

## The Original Meme

Ralph Wiggum is a character from The Simpsons known for his persistent but chaotic attempts at helping. The technique embraces this spirit: deterministic iteration that may seem chaotic but ultimately gets the job done.

The original Claude Code implementation became a meme in the AI development community for its simplicity and effectiveness. Learn more at [awesomeclaude.ai/ralph-wiggum](https://awesomeclaude.ai/ralph-wiggum).

---

*"Me fail English? That's unpossible!" - Ralph Wiggum*
