# Telegram Push Notifications

Send Telegram notifications when workflows complete or as part of other workflows.

## File Structure

This workflow consists of:
- `telegram-push.yml` - Main workflow definition with two workflows
- `scripts/send-telegram.sh` - Bash script for sending messages
- `.env.example` - Example environment variables
- `README.md` - Documentation

**Configuration**: Telegram credentials are stored in `.env` file in your project root (not in the workflow directory).

## Setup

### 1. Create a Telegram Bot

1. Open Telegram and search for [@BotFather](https://t.me/botfather)
2. Send `/newbot` command
3. Follow the instructions to create your bot
4. Copy the bot token (looks like `123456789:ABCdefGHIjklMNOpqrsTUVwxyz`)

### 2. Get Your Chat ID

1. Start a chat with your new bot
2. Send any message to the bot
3. Visit this URL in your browser (replace `YOUR_BOT_TOKEN` with your actual token):
   ```
   https://api.telegram.org/botYOUR_BOT_TOKEN/getUpdates
   ```
4. Look for `"chat":{"id":` in the response - this is your chat ID

### 3. Configure the Workflow

1. Add the following variables to your project's `.env` file (in the project root):
   ```bash
   TELEGRAM_BOT_TOKEN="123456789:ABCdefGHIjklMNOpqrsTUVwxyz"
   TELEGRAM_CHAT_ID="123456789"
   ```

2. If you don't have a `.env` file yet, create one:
   ```bash
   touch .env
   echo 'TELEGRAM_BOT_TOKEN="YOUR_BOT_TOKEN_HERE"' >> .env
   echo 'TELEGRAM_CHAT_ID="YOUR_CHAT_ID_HERE"' >> .env
   ```

3. **Important**: Make sure `.env` is in your `.gitignore` to keep credentials safe:
   ```bash
   echo ".env" >> .gitignore
   ```

## Usage

### Option 1: Send Telegram Notification (with task)

This workflow has a button trigger. When you click it:
1. Verifies that the prompt is not empty
2. Executes your current prompt
3. Waits for the agent to finish
4. Sends a Telegram notification with the task and result
5. Clears the prompt

**Use case**: Get notified when long-running tasks complete.

**Example**:
- Type: "Refactor the authentication module and add tests"
- Click the "Send Telegram Notification (with task)" button
- The agent will work on the task
- You'll receive a Telegram message when it's done

**Note**: If the prompt is empty when you trigger the workflow, it will skip execution (no notification will be sent).

### Option 2: Send Telegram Push

This is a utility workflow (no button) designed to be used as the last step in other workflows.

**Use case**: Send custom notifications from within workflows.

**How to use in other workflows**:

```yaml
Your Workflow:
  icon: 🚀
  menu: buttons
  actions:
    - name: Do Something
      prompt: 'Analyze the codebase'
      instantRun: true

    - name: Prepare Notification
      prompt: '✅ Analysis complete! Found 3 potential improvements.'

    - 'Send Telegram Push'  # Reference by name
```

Or by ID:
```yaml
actions:
  - name: Set message
    prompt: 'Task completed successfully!'
  - '5854f736-9198-4394-9dfe-telegram-push-02'
```

## Testing

Test your setup by running the script directly:

```bash
bash .supercode/workflows/telegram-push/scripts/send-telegram.sh "Hello from Supercode! 🚀"
```

You should receive a message in your Telegram chat.

## Troubleshooting

### ".env file not found"
Make sure `.env` exists in your project root directory

### "Invalid configuration"
Check that both `TELEGRAM_BOT_TOKEN` and `TELEGRAM_CHAT_ID` are properly set in your `.env` file

### "Failed to send Telegram notification"
- Verify your bot token is correct
- Make sure you've started a chat with your bot
- Check your chat ID is correct
- Ensure you have internet connectivity

## Advanced Usage

### Format Messages with Markdown

The script supports Telegram's Markdown formatting:

```yaml
prompt: |
  *Task Completed!* ✅

  _Project_: My Awesome App

  Changes:
  • Added authentication
  • Fixed bug in login
  • Updated tests

  Status: `SUCCESS`
```

### Send File Paths

```yaml
prompt:
  js: |
    `📝 Files modified:

    ${response.match(/\w+\.\w+/g).slice(0, 5).join('\n')}`
```

## Examples

### Daily Health Check with Notification

```yaml
Daily Health Check:
  icon: 🏥
  runEvery: '24h'
  actions:
    - name: Run Tests
      prompt:
        command: 'npm test 2>&1 | tail -20'

    - name: Prepare Report
      prompt:
        js: |
          (() => {
            const hasErrors = prompt.includes('FAIL') || prompt.includes('error');
            const emoji = hasErrors ? '❌' : '✅';
            return `${emoji} Daily Health Check\n\nTest Results:\n${prompt}`;
          })()

    - 'Send Telegram Push'
```

### Error Alert from CI

```yaml
CI Error Alert:
  icon: 🚨
  actions:
    - name: Check Build Status
      prompt:
        command: 'npm run build 2>&1 || echo "BUILD_FAILED"'

    - name: Send Alert
      if:
        type: js
        value: 'prompt.includes("BUILD_FAILED") || prompt.includes("error")'
      actions:
        - name: Format Error Message
          prompt:
            js: '`🚨 Build Failed!\n\nErrors:\n${prompt}`'

        - 'Send Telegram Push'
```
