#!/bin/bash

# Telegram notification script
# Usage: ./send-telegram.sh "Your message here"

# Load environment variables from .env file
ENV_FILE=".env"

# Check if .env file exists
if [ ! -f "$ENV_FILE" ]; then
    echo "❌ .env file not found at project root"
    echo "Please create it with the following variables:"
    echo 'TELEGRAM_BOT_TOKEN="YOUR_BOT_TOKEN"'
    echo 'TELEGRAM_CHAT_ID="YOUR_CHAT_ID"'
    exit 1
fi

# Source .env file
set -a
source "$ENV_FILE"
set +a

# Validate configuration
if [ -z "$TELEGRAM_BOT_TOKEN" ] || [ -z "$TELEGRAM_CHAT_ID" ]; then
    echo "❌ Invalid configuration. Please check your .env file."
    echo "Required variables: TELEGRAM_BOT_TOKEN, TELEGRAM_CHAT_ID"
    exit 1
fi

# Get message from argument or stdin
MESSAGE="$1"
if [ -z "$MESSAGE" ]; then
    MESSAGE=$(cat)
fi

# Escape message for JSON
MESSAGE=$(echo "$MESSAGE" | sed 's/\\/\\\\/g' | sed 's/"/\\"/g' | sed ':a;N;$!ba;s/\n/\\n/g')

# Send message to Telegram
RESPONSE=$(curl -s -X POST "https://api.telegram.org/bot${TELEGRAM_BOT_TOKEN}/sendMessage" \
    -H "Content-Type: application/json" \
    -d "{\"chat_id\":\"${TELEGRAM_CHAT_ID}\",\"text\":\"${MESSAGE}\",\"parse_mode\":\"Markdown\"}")

# Check if message was sent successfully
if echo "$RESPONSE" | grep -q '"ok":true'; then
    echo "✅ Telegram notification sent successfully"
    exit 0
else
    echo "❌ Failed to send Telegram notification"
    echo "$RESPONSE"
    exit 1
fi
