# Trello Task Processor Workflow

Automated workflow for processing Trello tasks: takes a task from Todo column, executes it with AI agent, generates summary, and moves it to Done.

## Features

-   ✅ Automatic task loading from Trello Todo
-   🧠 Intelligent analysis and task decomposition (SC:Architect)
-   📋 Step-by-step implementation plan creation
-   🔄 Cyclic execution with testing after each step
-   📝 Automatic work summary generation
-   ✅ Moving to Done with comment addition
-   🎨 Visual status indication (card colors)

## Installation

### 1. Get Trello API Keys

1. Get API Key: https://trello.com/app-key
2. Generate Token (click "Token" link on the page above)
3. Get Board ID:
    - Open your board in Trello
    - URL will be: `https://trello.com/b/BOARD_ID/board-name`
    - `BOARD_ID` is what you need

### 2. Configure Environment Variables

Copy `.env.example` from workflow directory to project root and fill in values:

```bash
# In project root create/update .env
cp .supercode/workflows/trello-task-processor/.env.example .env

# Edit .env and fill in your keys:
TRELLO_API_KEY=your_api_key_here
TRELLO_API_TOKEN=your_token_here
TRELLO_BOARD_ID=your_board_id_here
```

**Important:** The `.env` file must be in project root, not in workflow directory!

### 3. Ensure Trello Columns Exist

Workflow expects these columns:

-   **Todo** - for new tasks
-   **Done** - for completed tasks

Columns can contain counters in parentheses (e.g., "Done (15)") - workflow recognizes them.

## How It Works

### Workflow Structure

The workflow consists of two parts:

**1. Trello Task Processor** (main, with button):

```yaml
🔍 Load Task from Trello          # Load next task
❌ No Tasks Available              # If empty - exit
✅ Process Task                    # If available - process:
  📊 Parse Task Data               # JSON parsing
  🧠 Analyze & Decompose           # SC:Architect analysis
  📋 Create Implementation Plan    # Plan to trello-task-plan.md
  🛠️ Execute Implementation        # Start loop
  📝 Generate Task Summary         # Generate summary
  ✅ Complete Task in Trello       # Prepare data
  📤 Move to Done                  # Move + comment
  🧹 Cleanup                       # Remove temp files
```

**2. Trello Task Implementation Loop** (internal loop):

```yaml
📖 Read Implementation Plan        # Read plan
🔍 Check if More Steps            # Check for [ ] steps
  🚀 Implement Next Step          # Take first [ ], implement
    Tests Must Pass               # Validation
    Loop recursion                # Next step
```

### Task Lifecycle

1. **Task in Todo** - regular card
2. **Taken to work** - `[🤖 AI Working] Task name` (yellow)
3. **In progress** - AI analyzes, plans, implements
4. **Completed** - moved to Done (green) with comment

### Visual Indicators

-   **Yellow card** + prefix `[🤖 AI Working]` - task in progress by AI
-   **Green card** in Done - task completed
-   **Comment** with ✅ emoji - work summary

## Usage

### Basic Usage

1. Create tasks in "Todo" column in Trello
2. Click "Trello Task Processor" button in Supercode
3. Workflow will:
    - Take first available task
    - Analyze and plan
    - Implement step by step with testing
    - Generate summary
    - Move to Done

### Task Format in Trello

**Card title:**

```
Add user authentication
```

**Description (optional):**

```
Implement JWT-based authentication with:
- Login endpoint
- Token validation middleware
- Password hashing
- Tests for auth flow
```

The more detailed the description, the better AI understands the task.

### Automatic Processing Loop

To process tasks automatically, you can:

1. **Option 1:** Use `runEvery` (requires running Supercode):

```yaml
Trello Task Processor:
    runEvery: '5m' # Every 5 minutes
    # ...
```

2. **Option 2:** External cron job:

```bash
# Crontab entry (every 10 minutes)
*/10 * * * * cd /path/to/project && supercode run "Trello Task Processor"
```

## Task Examples

### Simple Task

**Title:** Fix login button styling **Description:** The login button on /auth/login page has wrong padding and color

**What AI will do:**

1. Find button component
2. Fix styles
3. Check in other places
4. Run tests
5. Summary: "Fixed login button padding (12px → 16px) and color (blue → brand-primary) in LoginButton.tsx. Verified consistent styling across app."

### Complex Task

**Title:** Implement real-time notifications **Description:**

```
Add WebSocket-based notifications:
- Server-side WebSocket handler
- Client-side connection manager
- Notification toast component
- Persistence layer
- Tests for connection handling
```

**What AI will do:**

1. Decompose into ~7 steps
2. Create WebSocket server
3. Implement client
4. Add UI component
5. Implement persistence
6. Write tests
7. Each step tested separately
8. Summary: "Implemented WebSocket notifications system with server handler (ws.ts), client manager (NotificationManager.ts), toast UI (NotificationToast.tsx), and SQLite persistence. All tests passing, coverage 85%."

## Benefits

1. **Fully automated** - from taking task to moving to Done
2. **Safe** - each step tested before proceeding to next
3. **Transparent** - progress visible in Trello (colors, prefixes)
4. **Documented** - summary in comments
5. **Scalable** - can process task queue
6. **Intelligent** - uses SC:Architect for analysis

## Troubleshooting

### Workflow doesn't find tasks

-   Check that column is named exactly "Todo" (case sensitive)
-   Ensure Todo has cards without prefix `[🤖 AI Working]`
-   Verify TRELLO_BOARD_ID in .env

### Error "List not found"

-   Check that TRELLO_BOARD_ID is correct
-   Ensure API Key and Token are valid
-   Verify token has access to this board

### Task doesn't move to Done

-   Ensure "Done" column exists
-   Check workflow logs for errors
-   Tests might have failed and workflow stopped

### Tests Must Pass looped infinitely

-   Check that tests can actually pass
-   Workflow stops after 20 iterations
-   Can manually complete and refine

## Customization

### Change Column Names

Edit `scripts/load-task.ts` and `scripts/complete-task.ts`:

```typescript
const TODO_LIST_NAME = 'Backlog'; // instead of "Todo"
const DONE_LIST_NAME = 'Completed'; // instead of "Done"
```

### Add Additional Statuses

You can create workflows for other columns:

```yaml
Trello Backlog Processor:
    # Takes from Backlog, analyzes, moves to Todo

Trello Review Processor:
    # Takes from Review, checks, moves to Done
```

### Change Card Colors

In `load-task.ts` and `complete-task.ts`:

```typescript
'yellow'; // In progress
'green'; // Completed
'red'; // Failed
'blue'; // Review needed
```

## Architecture

```
trello-task-processor/
├── scripts/
│   ├── trello-utils.ts       # Trello API utilities
│   ├── load-task.ts          # Load next task from Todo
│   ├── get-current-task.ts   # Find in-progress task by prefix
│   └── complete-task.ts      # Move to Done + add summary
├── .env.example              # Environment variables template
├── trello-task-processor.yml # Main workflow definition
└── README.md                 # This file
```

**Configuration:**

-   `.env` - Environment variables (in project root, not committed)
-   `.env.example` - Configuration template

**Temporary files** (created during execution):

-   `.trello-task-plan.md` - Implementation plan with checkboxes

## Best Practices

1. **Detailed task descriptions** - more context = better result
2. **Atomic tasks** - one task = one feature/bugfix
3. **Have tests** - workflow relies on tests for validation
4. **Code review** - don't forget to review code after AI
5. **Commits** - commit after each completed task

## Integration with Other Workflows

Can be combined with other workflows:

```yaml
My Custom Workflow:
    actions:
        - 'Trello Task Processor' # Process task from Trello
        - name: Deploy
          prompt:
              command: 'npm run deploy'
```

## Security

-   ⚠️ **Don't commit .env file** with API keys
-   ⚠️ **Limit token permissions** to only necessary boards
-   ⚠️ **Review code** after AI execution
-   ⚠️ **Test on dev board** before production

---

_Powered by Supercode Workflows 2.0_
