import 'dotenv/config';

const API_KEY = process.env.TRELLO_API_KEY;
const API_TOKEN = process.env.TRELLO_API_TOKEN;
const BOARD_ID = process.env.TRELLO_BOARD_ID;

if (!API_KEY || !API_TOKEN) {
	throw new Error('TRELLO_API_KEY and TRELLO_API_TOKEN are required in .env file');
}

if (!BOARD_ID) {
	throw new Error('TRELLO_BOARD_ID is required in .env file');
}

const cleanObject = (obj: any): any => {
	return Object.fromEntries(Object.entries(obj).filter(([_, value]) => value !== undefined));
};

const jsonRequest = (method: string, url: string, body?: any) => {
	const qs = new URLSearchParams({
		key: API_KEY!,
		token: API_TOKEN!,
	});
	return fetch(url + '?' + qs.toString(), {
		method: method,
		headers: cleanObject({
			'Accept': 'application/json',
			'Content-Type': body ? 'application/json' : undefined,
		}),
		body: body ? JSON.stringify(body) : undefined,
	}).then(response => {
		if (!response.ok) {
			return response.text().then(text => {
				throw new Error(`Trello API error: ${response.status} ${text}`);
			});
		}
		return response.json();
	});
};

const get = (url: string) => jsonRequest('GET', url);
const post = (url: string, body: any) => jsonRequest('POST', url, body);
const put = (url: string, body: any) => jsonRequest('PUT', url, body);

export const getLists = async (boardId: string) => {
	return get(`https://api.trello.com/1/boards/${boardId}/lists`);
};

export const getListByName = async (
	boardId: string,
	listName: string,
): Promise<{ id: string; name: string } | null> => {
	const lists = await getLists(boardId);
	const matchingList = lists.find((list: any) => {
		if (list.name === listName) return true;
		if (list.name.startsWith(listName)) {
			const nextChar = list.name[listName.length];
			return nextChar === ' ' || nextChar === '(';
		}
		return false;
	});
	return matchingList ? { id: matchingList.id, name: matchingList.name } : null;
};

export const getCards = async (listId: string) => {
	return get(`https://api.trello.com/1/lists/${listId}/cards`);
};

export const updateCard = async (
	cardId: string,
	newName?: string,
	newDescription?: string,
	color?: 'pink' | 'yellow' | 'lime' | 'blue' | 'black' | 'orange' | 'red' | 'purple' | 'sky' | 'green',
) => {
	return put(
		`https://api.trello.com/1/cards/${cardId}`,
		cleanObject({
			name: newName,
			desc: newDescription,
			cover: cleanObject({
				color: color,
			}),
		}),
	);
};

export const moveCard = async (cardId: string, listId: string) => {
	return put(`https://api.trello.com/1/cards/${cardId}`, {
		idList: listId,
	});
};

export const addComment = async (cardId: string, text: string) => {
	return post(`https://api.trello.com/1/cards/${cardId}/actions/comments`, {
		text,
	});
};

export const getCard = async (cardId: string) => {
	return get(`https://api.trello.com/1/cards/${cardId}`);
};

export { BOARD_ID };
